"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
var NotificationsService_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NotificationsService = void 0;
const common_1 = require("@nestjs/common");
const mongoose_1 = require("@nestjs/mongoose");
const mongoose_2 = require("mongoose");
const webpush = require("web-push");
const push_subscription_schema_1 = require("./schemas/push-subscription.schema");
const notification_schema_1 = require("./schemas/notification.schema");
const delivery_log_schema_1 = require("./schemas/delivery-log.schema");
let NotificationsService = NotificationsService_1 = class NotificationsService {
    constructor(pushSubscriptionModel, notificationModel, deliveryLogModel) {
        this.pushSubscriptionModel = pushSubscriptionModel;
        this.notificationModel = notificationModel;
        this.deliveryLogModel = deliveryLogModel;
        this.logger = new common_1.Logger(NotificationsService_1.name);
        const vapidPublicKey = process.env.VAPID_PUBLIC_KEY;
        const vapidPrivateKey = process.env.VAPID_PRIVATE_KEY;
        const vapidSubject = process.env.VAPID_SUBJECT || 'mailto:admin@nezampezeshki.ir';
        if (vapidPublicKey && vapidPrivateKey) {
            webpush.setVapidDetails(vapidSubject, vapidPublicKey, vapidPrivateKey);
            this.logger.log('Web Push VAPID configured successfully');
        }
        else {
            this.logger.warn('VAPID keys not configured. Push notifications will not work. Generate keys with: npx web-push generate-vapid-keys');
        }
    }
    async subscribe(userId, subscribeDto) {
        try {
            const existing = await this.pushSubscriptionModel.findOne({
                endpoint: subscribeDto.endpoint,
            });
            if (existing) {
                existing.userId = userId;
                existing.p256dh = subscribeDto.p256dh;
                existing.auth = subscribeDto.auth;
                existing.userAgent = subscribeDto.userAgent;
                existing.isActive = true;
                existing.lastUsed = new Date();
                await existing.save();
                return existing;
            }
            const subscription = new this.pushSubscriptionModel({
                userId,
                endpoint: subscribeDto.endpoint,
                p256dh: subscribeDto.p256dh,
                auth: subscribeDto.auth,
                userAgent: subscribeDto.userAgent,
                lastUsed: new Date(),
            });
            return await subscription.save();
        }
        catch (error) {
            this.logger.error('Error subscribing to push notifications', error);
            throw error;
        }
    }
    async unsubscribe(userId, endpoint) {
        try {
            const result = await this.pushSubscriptionModel.deleteOne({
                userId,
                endpoint,
            });
            return result.deletedCount > 0;
        }
        catch (error) {
            this.logger.error('Error unsubscribing from push notifications', error);
            throw error;
        }
    }
    async getUserSubscriptions(userId) {
        return await this.pushSubscriptionModel.find({ userId, isActive: true });
    }
    async getSubscriptionCount() {
        return await this.pushSubscriptionModel.countDocuments({ isActive: true });
    }
    async createNotification(createDto) {
        const notification = new this.notificationModel(createDto);
        const saved = await notification.save();
        this.sendPushToAll(saved).catch(err => {
            this.logger.error('Error sending push notifications', err);
        });
        return saved;
    }
    async updateNotification(id, updateDto) {
        return await this.notificationModel.findByIdAndUpdate(id, updateDto, { new: true });
    }
    async deleteNotification(id) {
        return await this.notificationModel.findByIdAndDelete(id);
    }
    async getAllNotifications(page = 1, limit = 20) {
        const skip = (page - 1) * limit;
        const [notifications, total] = await Promise.all([
            this.notificationModel.find().sort({ createdAt: -1 }).skip(skip).limit(limit),
            this.notificationModel.countDocuments(),
        ]);
        return { notifications, total, page, totalPages: Math.ceil(total / limit) };
    }
    async getNotification(id) {
        return await this.notificationModel.findById(id);
    }
    async sendPushToAll(notification) {
        const subscriptions = await this.pushSubscriptionModel.find({ isActive: true });
        this.logger.log(`Sending push notification to ${subscriptions.length} subscribers`);
        let sentCount = 0;
        let failedCount = 0;
        const payload = JSON.stringify({
            title: notification.title,
            body: notification.text,
            icon: notification.icon || '/assets/img/logo-192.png',
            badge: '/assets/img/logo-96.png',
            image: notification.image,
            data: {
                url: notification.link,
                linkText: notification.linkText,
                notificationId: notification._id,
            },
        });
        for (const subscription of subscriptions) {
            try {
                const pushSubscription = {
                    endpoint: subscription.endpoint,
                    keys: {
                        p256dh: subscription.p256dh,
                        auth: subscription.auth,
                    },
                };
                const result = await webpush.sendNotification(pushSubscription, payload);
                await this.logDelivery(notification._id, subscription._id, subscription.endpoint, true, result.statusCode);
                sentCount++;
                subscription.lastUsed = new Date();
                await subscription.save();
            }
            catch (error) {
                this.logger.error(`Failed to send push to ${subscription.endpoint}`, error);
                await this.logDelivery(notification._id, subscription._id, subscription.endpoint, false, error.statusCode, error.message);
                failedCount++;
                if (error.statusCode === 410) {
                    subscription.isActive = false;
                    await subscription.save();
                }
            }
        }
        notification.sentCount = sentCount;
        notification.failedCount = failedCount;
        await notification.save();
        this.logger.log(`Push notification sent: ${sentCount} successful, ${failedCount} failed`);
        return { sentCount, failedCount };
    }
    async logDelivery(notificationId, subscriptionId, endpoint, success, statusCode, error) {
        try {
            const log = new this.deliveryLogModel({
                notificationId,
                subscriptionId,
                endpoint,
                success,
                statusCode,
                error,
            });
            await log.save();
        }
        catch (err) {
            this.logger.error('Error logging delivery', err);
        }
    }
    async getDeliveryLogs(notificationId) {
        return await this.deliveryLogModel
            .find({ notificationId })
            .sort({ createdAt: -1 })
            .limit(1000);
    }
    getVapidPublicKey() {
        return process.env.VAPID_PUBLIC_KEY;
    }
};
exports.NotificationsService = NotificationsService;
exports.NotificationsService = NotificationsService = NotificationsService_1 = __decorate([
    (0, common_1.Injectable)(),
    __param(0, (0, mongoose_1.InjectModel)(push_subscription_schema_1.PushSubscription.name)),
    __param(1, (0, mongoose_1.InjectModel)(notification_schema_1.Notification.name)),
    __param(2, (0, mongoose_1.InjectModel)(delivery_log_schema_1.DeliveryLog.name)),
    __metadata("design:paramtypes", [mongoose_2.Model,
        mongoose_2.Model,
        mongoose_2.Model])
], NotificationsService);
//# sourceMappingURL=notifications.service.js.map