"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContactService = void 0;
const common_1 = require("@nestjs/common");
const mongoose_1 = require("@nestjs/mongoose");
const mongoose_2 = require("mongoose");
const contact_message_schema_1 = require("./schemas/contact-message.schema");
const sms_service_1 = require("../../shared/sms.service");
const crypto_1 = require("crypto");
let ContactService = class ContactService {
    constructor(contactModel, smsService) {
        this.contactModel = contactModel;
        this.smsService = smsService;
        this.rateLimitStore = new Map();
        this.numericCaptchaStore = new Map();
        this.numericCaptchaTTL = 3 * 60 * 1000;
    }
    async submitMessage(dto, clientIp) {
        const identifier = clientIp || dto.email || dto.phone || 'unknown';
        this.checkRateLimit(identifier);
        const message = new this.contactModel({
            fullName: dto.fullName,
            email: dto.email,
            phone: dto.phone,
            subject: dto.subject,
            message: dto.message,
            createdAt: new Date(),
            read: false,
        });
        const savedMessage = await message.save();
        await this.notifyAdminsViaSms(dto);
        return savedMessage;
    }
    verifyNumericCaptcha(token, code) {
        this.purgeExpiredNumericCaptchas();
        const entry = this.numericCaptchaStore.get(token);
        if (!entry) {
            throw new common_1.BadRequestException('کد امنیتی منقضی شده یا معتبر نیست. لطفا دوباره تلاش کنید.');
        }
        if (entry.code !== code) {
            throw new common_1.BadRequestException('کد امنیتی اشتباه وارد شده است. لطفا دوباره سعی کنید.');
        }
        this.numericCaptchaStore.delete(token);
    }
    generateNumericCaptcha(length = 5) {
        this.purgeExpiredNumericCaptchas();
        const token = (0, crypto_1.randomUUID)();
        const code = Array.from({ length })
            .map(() => Math.floor(Math.random() * 10))
            .join("");
        const expiresAt = Date.now() + this.numericCaptchaTTL;
        this.numericCaptchaStore.set(token, { code, expiresAt });
        return {
            token,
            challenge: code,
            expiresAt,
        };
    }
    purgeExpiredNumericCaptchas() {
        const now = Date.now();
        for (const [key, entry] of this.numericCaptchaStore) {
            if (entry.expiresAt < now) {
                this.numericCaptchaStore.delete(key);
            }
        }
    }
    checkRateLimit(identifier) {
        const now = Date.now();
        const ONE_HOUR_MS = 60 * 60 * 1000;
        const ONE_MINUTE_MS = 60 * 1000;
        const MAX_SUBMISSIONS = 5;
        const existing = this.rateLimitStore.get(identifier);
        if (existing && existing.lastSubmission && now - existing.lastSubmission < ONE_MINUTE_MS) {
            throw new common_1.BadRequestException('لطفا حداقل یک دقیقه بین ارسال‌های خود فاصله دهید.');
        }
        if (existing) {
            if (now < existing.resetTime) {
                if (existing.count >= MAX_SUBMISSIONS) {
                    const minutesRemaining = Math.ceil((existing.resetTime - now) / 60000);
                    throw new common_1.BadRequestException(`بیش از حد درخواست ارسال شده است. لطفا ${minutesRemaining} دقیقه دیگر دوباره سعی کنید`);
                }
                existing.count++;
                existing.lastSubmission = now;
            }
            else {
                this.rateLimitStore.set(identifier, { count: 1, resetTime: now + ONE_HOUR_MS, lastSubmission: now });
            }
        }
        else {
            this.rateLimitStore.set(identifier, { count: 1, resetTime: now + ONE_HOUR_MS, lastSubmission: now });
        }
    }
    async notifyAdminsViaSms(contactData) {
        try {
            const adminPhones = process.env.ADMIN_Phone?.split(',') || [];
            if (!adminPhones || adminPhones.length === 0) {
                return;
            }
            const smsMessage = `[تماس جدید]\n👤 ${contactData.fullName}\n📱 ${contactData.phone || contactData.email}\n📌 ${contactData.subject}`;
            for (const phone of adminPhones) {
                const cleanPhone = phone.trim();
                if (cleanPhone) {
                    this.smsService.sendSms(cleanPhone, smsMessage).catch((error) => {
                        console.error(`Failed to send SMS to admin ${cleanPhone}:`, error);
                    });
                }
            }
        }
        catch (error) {
            console.error('Error notifying admins via SMS:', error);
        }
    }
    async getMessages(page = 1, limit = 10) {
        const skip = (page - 1) * limit;
        const messages = await this.contactModel
            .find()
            .sort({ createdAt: -1 })
            .skip(skip)
            .limit(limit)
            .lean()
            .exec();
        const total = await this.contactModel.countDocuments();
        return {
            data: messages,
            total,
            page,
            limit,
            hasMore: page * limit < total,
        };
    }
    async markAsRead(messageId) {
        return this.contactModel.findByIdAndUpdate(messageId, { read: true }, { new: true });
    }
    async deleteMessage(messageId) {
        return this.contactModel.findByIdAndDelete(messageId);
    }
};
exports.ContactService = ContactService;
exports.ContactService = ContactService = __decorate([
    (0, common_1.Injectable)(),
    __param(0, (0, mongoose_1.InjectModel)(contact_message_schema_1.ContactMessage.name)),
    __metadata("design:paramtypes", [mongoose_2.Model,
        sms_service_1.SmsService])
], ContactService);
//# sourceMappingURL=contact.service.js.map