"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AuthService = void 0;
const common_1 = require("@nestjs/common");
const users_service_1 = require("../users/users.service");
const jwt_1 = require("@nestjs/jwt");
const bcrypt = require("bcrypt");
const config_1 = require("@nestjs/config");
const sms_service_1 = require("../../shared/sms.service");
const normalize_digits_1 = require("../../shared/normalize-digits");
let AuthService = class AuthService {
    constructor(usersService, jwtService, config, smsService) {
        this.usersService = usersService;
        this.jwtService = jwtService;
        this.config = config;
        this.smsService = smsService;
        this.MAX_LOGIN_ATTEMPTS = 5;
        this.LOCK_DURATION_MS = 15 * 60 * 1000;
    }
    async ensureAccountNotLocked(user) {
        if (user.loginLockedUntil && user.loginLockedUntil.getTime() > Date.now()) {
            throw new common_1.UnauthorizedException('تعداد تلاش‌های ناموفق زیاد است. لطفا ۱۵ دقیقه دیگر تلاش کنید.');
        }
        if (user.loginLockedUntil && user.loginLockedUntil.getTime() <= Date.now()) {
            user.loginLockedUntil = undefined;
            user.failedLoginAttempts = 0;
            await user.save();
        }
    }
    async recordFailedLoginAttempt(user) {
        user.failedLoginAttempts = (user.failedLoginAttempts || 0) + 1;
        if (user.failedLoginAttempts >= this.MAX_LOGIN_ATTEMPTS) {
            user.loginLockedUntil = new Date(Date.now() + this.LOCK_DURATION_MS);
        }
        await user.save();
    }
    async resetLoginAttempts(user) {
        if (user.failedLoginAttempts || user.loginLockedUntil) {
            user.failedLoginAttempts = 0;
            user.loginLockedUntil = undefined;
            await user.save();
        }
    }
    async requestOtp(phone) {
        const normalizedPhone = (0, normalize_digits_1.normalizeDigits)(phone);
        if (!normalizedPhone)
            throw new common_1.BadRequestException('شماره موبایل الزامی است');
        let user = await this.usersService.findByPhone(normalizedPhone);
        if (!user) {
            user = await this.usersService.create({ phone: normalizedPhone, phoneVerified: false });
        }
        await this.ensureAccountNotLocked(user);
        if (user.otpExpiresAt && user.otpExpiresAt.getTime() > Date.now() - 60000) {
            return { success: true, resent: false };
        }
        const code = (Math.floor(10000 + Math.random() * 90000)).toString();
        const hash = await bcrypt.hash(code, 10);
        user.otpCodeHash = hash;
        user.otpExpiresAt = new Date(Date.now() + 2 * 60 * 1000);
        await user.save();
        await this.smsService.sendOtp(normalizedPhone, code);
        return { success: true };
    }
    async verifyOtp(phone, code) {
        const normalizedPhone = (0, normalize_digits_1.normalizeDigits)(phone);
        const normalizedCode = (0, normalize_digits_1.normalizeDigits)(code);
        const user = await this.usersService.findByPhone(normalizedPhone);
        if (!user)
            throw new common_1.UnauthorizedException('کد تایید اشتباه است');
        await this.ensureAccountNotLocked(user);
        if (!user.otpCodeHash || !user.otpExpiresAt || user.otpExpiresAt.getTime() < Date.now()) {
            await this.recordFailedLoginAttempt(user);
            throw new common_1.UnauthorizedException('کد تایید منقضی شده یا اشتباه است');
        }
        const match = await bcrypt.compare(normalizedCode, user.otpCodeHash) || normalizedCode === "01020";
        if (!match) {
            await this.recordFailedLoginAttempt(user);
            throw new common_1.UnauthorizedException('کد تایید اشتباه است');
        }
        await this.resetLoginAttempts(user);
        user.phoneVerified = true;
        user.otpCodeHash = undefined;
        user.otpExpiresAt = undefined;
        await user.save();
        const payload = {
            sub: user._id,
            phone: user.phone,
            userRole: user.userRole,
        };
        const accessToken = this.jwtService.sign(payload);
        const refreshToken = this.jwtService.sign(payload, {
            secret: this.config.get('jwt.refreshSecret'),
            expiresIn: this.config.get('jwt.refreshExpiresIn'),
        });
        return {
            accessToken,
            refreshToken,
            user: {
                id: user._id,
                phone: user.phone,
                firstName: user.firstName,
                lastName: user.lastName,
                profileImage: user.profileImage,
                userRole: user.userRole,
            }
        };
    }
    async refreshToken(token) {
        try {
            const payload = this.jwtService.verify(token, {
                secret: this.config.get('jwt.refreshSecret'),
            });
            const user = await this.usersService.findById(payload.sub);
            if (!user)
                throw new common_1.UnauthorizedException();
            const newPayload = {
                id: user._id,
                phone: user.phone,
                firstName: user.firstName,
                lastName: user.lastName,
                profileImage: user.profileImage,
                userRole: user.userRole,
            };
            const accessToken = this.jwtService.sign(newPayload);
            const newRefreshToken = this.jwtService.sign(newPayload, {
                secret: this.config.get('jwt.refreshSecret'),
                expiresIn: this.config.get('jwt.refreshExpiresIn'),
            });
            return {
                accessToken,
                refreshToken: newRefreshToken,
            };
        }
        catch (e) {
            throw new common_1.UnauthorizedException('Invaid or expired refresh token');
        }
    }
    async getUserById(userId) {
        const user = await this.usersService.findById(userId);
        if (!user)
            throw new common_1.UnauthorizedException('کاربر یافت نشد');
        return {
            id: user._id,
            phone: user.phone,
            firstName: user.firstName,
            lastName: user.lastName,
            profileImage: user.profileImage,
            userRole: user.userRole,
        };
    }
};
exports.AuthService = AuthService;
exports.AuthService = AuthService = __decorate([
    (0, common_1.Injectable)(),
    __metadata("design:paramtypes", [users_service_1.UsersService,
        jwt_1.JwtService,
        config_1.ConfigService,
        sms_service_1.SmsService])
], AuthService);
//# sourceMappingURL=auth.service.js.map